<?php
/**
 * ===================================================================================
 * Reply to Message Page (Mobile Responsive)
 * ===================================================================================
 *
 * File: reply_message.php
 * Location: /homework_portal/teacher/reply_message.php
 *
 * Purpose:
 * 1. Provides a responsive form for a teacher to reply to a specific admin message.
 * 2. Fetches the original message for context.
 * 3. Saves the reply as a new message linked to the original thread.
 */

require_once '../config.php';

// --- Security Check & Initial Setup ---
if (!isset($_SESSION['teacher_id'])) {
    header('Location: login.php');
    exit();
}

$teacher_id = $_SESSION['teacher_id'];
$teacher_full_name = e($_SESSION['teacher_full_name'] ?? 'Teacher');
$parent_message_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];

if ($parent_message_id <= 0) {
    header('Location: inbox.php');
    exit();
}

// --- Fetch the original message to reply to ---
try {
    // Ensure the teacher can only reply to messages sent to them or in their thread
    $stmt = $pdo->prepare("SELECT * FROM teacher_messages WHERE id = ? AND teacher_id = ?");
    $stmt->execute([$parent_message_id, $teacher_id]);
    $original_message = $stmt->fetch();

    if (!$original_message) {
        // This could be a reply to a reply, find the root message of the thread.
        $thread_stmt = $pdo->prepare("SELECT * FROM teacher_messages WHERE parent_message_id = ? AND teacher_id = ? ORDER BY created_at ASC LIMIT 1");
        $thread_stmt->execute([$parent_message_id, $teacher_id]);
        $original_message = $thread_stmt->fetch();
    }
    
    if (!$original_message) {
        header('Location: inbox.php?error=Message not found.');
        exit();
    }

    // Determine the root of the thread
    $thread_id = $original_message['parent_message_id'] ?? $original_message['id'];


    // --- Handle Form Submission ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_reply'])) {
        $title = "Re: " . $original_message['title']; // Prepend "Re:" to the original title
        $message_content = trim($_POST['message'] ?? '');
        
        if (empty($message_content)) {
            $errors[] = "Your reply cannot be empty.";
        }

        if (empty($errors)) {
            $sql = "INSERT INTO teacher_messages (sender_type, admin_id, teacher_id, title, message, parent_message_id) VALUES (?, ?, ?, ?, ?, ?)";
            $pdo->prepare($sql)->execute(['teacher', $original_message['admin_id'], $teacher_id, $title, $message_content, $thread_id]);
            
            $_SESSION['success_message'] = "Your reply has been sent successfully!";
            header("Location: inbox.php");
            exit();
        }
    }
    
    // Fetch counts for sidebar badges
    $unread_messages_count = $pdo->query("SELECT COUNT(id) FROM teacher_messages WHERE teacher_id = {$teacher_id} AND is_read = 0 AND sender_type = 'admin'")->fetchColumn();
    $new_submission_count = $pdo->query("SELECT COUNT(sub.id) FROM submissions sub JOIN assignments a ON sub.assignment_id = a.id LEFT JOIN teacher_seen_submissions tss ON sub.id = tss.submission_id AND tss.teacher_id = {$teacher_id} WHERE a.teacher_id = {$teacher_id} AND tss.submission_id IS NULL")->fetchColumn();


} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reply to Message - Teacher Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f9ff; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .sidebar-link { display: flex; align-items: center; padding: 0.75rem 1rem; border-radius: 0.5rem; transition: background-color 0.2s, color 0.2s; font-weight: 500;}
        .sidebar-link:hover, .sidebar-link.active { background-color: #e0f2fe; color: #0284c7; font-weight: 600; }
        .notification-badge { display: inline-flex; align-items: center; justify-content: center; min-width: 22px; height: 22px; padding: 0 4px; border-radius: 9999px; background-color: #ef4444; color: white; font-size: 0.75rem; font-weight: bold; }
    </style>
</head>
<body class="flex flex-col min-h-screen">
    <div class="relative min-h-screen md:flex">
        <!-- Mobile menu button -->
        <div class="md:hidden flex justify-between items-center bg-white p-4 shadow-md">
             <a href="dashboard.php">
                <h1 class="text-3xl font-bold text-red-600 font-sans">TTTEDDDEL</h1>
            </a>
            <button id="mobile-menu-button" class="text-gray-500 hover:text-blue-600 focus:outline-none">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <aside id="sidebar" class="w-64 bg-white shadow-md absolute inset-y-0 left-0 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out z-30">
            <div class="p-4 border-b text-center">
                <a href="dashboard.php">
                    <h1 class="text-4xl font-bold text-red-600 tracking-widest font-sans">TTTEDDDEL</h1>
                    <p class="text-xs font-semibold text-blue-800 mt-1">Teacher-Student Interactive System</p>
                    <p class="text-xs text-blue-800">by Ele Infocomm Technologies</p>
                </a>
            </div>
            <nav class="p-4 space-y-2">
                <a href="dashboard.php" class="sidebar-link"><span>Dashboard</span></a>
                <a href="inbox.php" class="sidebar-link active flex justify-between items-center">
                    <span>Inbox</span>
                    <?php if ($unread_messages_count > 0): ?><span class="notification-badge"><?php echo $unread_messages_count; ?></span><?php endif; ?>
                </a>
                <a href="manage_students.php" class="sidebar-link"><span>My Students</span></a>
                <a href="post_material.php" class="sidebar-link"><span>Post Material</span></a>
                <a href="my_materials.php" class="sidebar-link"><span>My Posted Materials</span></a>
                <a href="view_submissions.php" class="sidebar-link flex justify-between items-center">
                    <span>View Submissions</span>
                     <?php if ($new_submission_count > 0): ?><span class="notification-badge"><?php echo $new_submission_count; ?></span><?php endif; ?>
                </a>
            </nav>
        </aside>

        <!-- Main Content Area -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm border-b hidden md:flex"><div class="flex items-center justify-end p-4 w-full"><span class="text-gray-600 mr-4">Welcome, <?php echo $teacher_full_name; ?>!</span><a href="../logout.php" class="ml-2 py-2 px-4 bg-teal-600 text-white font-semibold rounded-lg shadow-md hover:bg-teal-700">Logout</a></div></header>
            
            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 md:p-6">
                <div class="mb-6">
                    <a href="inbox.php" class="inline-flex items-center text-teal-600 hover:text-teal-800 font-medium">
                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg>
                        Back to Inbox
                    </a>
                </div>

                <div class="bg-white p-6 rounded-lg shadow-lg">
                    <h1 class="text-2xl font-bold text-gray-800 mb-6 font-brand">Reply to Message</h1>

                    <!-- Original Message Context -->
                    <div class="mb-6 bg-gray-50 p-4 rounded-lg border">
                        <p class="text-sm font-semibold text-gray-700">In reply to:</p>
                        <h4 class="font-bold text-gray-900"><?php echo e($original_message['title']); ?></h4>
                        <p class="mt-2 text-sm text-gray-600 border-l-2 border-gray-300 pl-3">"<?php echo nl2br(e($original_message['message'])); ?>"</p>
                    </div>

                    <?php if (!empty($errors)): ?><div class="p-4 mb-4 text-sm text-red-700 bg-red-100 rounded-lg"><ul class="list-disc list-inside"><?php foreach ($errors as $e) echo "<li>" . e($e) . "</li>"; ?></ul></div><?php endif; ?>

                    <form action="reply_message.php?id=<?php echo $parent_message_id; ?>" method="POST" class="space-y-6">
                        <div>
                            <label for="message" class="block text-sm font-medium text-gray-700">Your Reply</label>
                            <textarea name="message" id="message" rows="8" class="mt-1 block w-full p-2 border border-gray-300 rounded-md" required></textarea>
                        </div>
                        <div class="flex justify-end pt-4 border-t">
                            <button type="submit" name="send_reply" class="w-full sm:w-auto px-6 py-2 bg-teal-600 text-white font-semibold rounded-lg shadow hover:bg-teal-700">Send Reply</button>
                        </div>
                    </form>
                </div>
            </main>
        </div>
    </div>
    <footer class="text-center py-6 bg-gray-100 border-t mt-auto">
        <p class="text-xs text-gray-500">Built By Ele InfoComm Technologies- Kenya | <a href="http://web.e-ict.co.ke" target="_blank" class="hover:underline">web.e-ict.co.ke</a> | <a href="mailto:info@e-ict.co.ke" class="hover:underline">info@e-ict.co.ke</a> | +254202000121 | +254 722 215810 | 2025 All right reserved.</p>
    </footer>

    <script>
        const btn = document.getElementById('mobile-menu-button');
        const sidebar = document.getElementById('sidebar');

        btn.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });
    </script>
</body>
</html>
